import random, time

DIFFICULTY = 40 # Liczba losowych przesunięć klocków wykonywana na początku programu
SIZE = 4 # Plansza ma wymiary SIZE x SIZE
random.seed(1) # Możliwość wyboru, która łamigłówka ma zostać rozwiązana

BLANK = 0
UP = 'w górę'
DOWN = 'w dół'
LEFT = 'w lewo'
RIGHT = 'w prawo'


def displayBoard(board):
    """Funkcja wyświetla klocki przechowywane w argumencie `board` na ekranie"""
    for y in range(SIZE): # Przejście przez każdy wiersz
        for x in range(SIZE): # Przejście przez każdą kolumnę
            if board[y * SIZE + x] == BLANK:
                print('__ ', end='') # Wyświetlenie pustego miejsca
            else:
                print(str(board[y * SIZE + x]).rjust(2) + ' ', end='')
        print() # Dodanie znaku nowej linii na końcu każdego wiersza


def getNewBoard():
    """Funkcja zwraca listę reprezentującą nową układankę"""
    board = []
    for i in range(1, SIZE * SIZE):
        board.append(i)
    board.append(BLANK)
    return board


def findBlankSpace(board):
    """Funkcja zwraca listę [x, y] zawierającą współrzędne pustego obszaru"""
    for x in range(SIZE):
        for y in range(SIZE):
            if board[y * SIZE + x] == BLANK:
                return [x, y]


def makeMove(board, move):
    """Funkcja przeprowadza ruch `move` i modyfikuje w miejscu zawartość `board`"""
    bx, by = findBlankSpace(board)
    blankIndex = by * SIZE + bx

    if move == UP:
        tileIndex = (by + 1) * SIZE + bx
    elif move == LEFT:
        tileIndex = by * SIZE + (bx + 1)
    elif move == DOWN:
        tileIndex = (by - 1) * SIZE + bx
    elif move == RIGHT:
        tileIndex = by * SIZE + (bx - 1)

    # Zamiana miejscami klocków kblankIndex i tileIndex:
    board[blankIndex], board[tileIndex] = board[tileIndex], board[blankIndex]


def undoMove(board, move):
    """Funkcja wykonuje ruch przeciwny do `move`, aby usunąć efekty jego wykonania wprowadzone do `board`"""
    if move == UP:
        makeMove(board, DOWN)
    elif move == DOWN:
        makeMove(board, UP)
    elif move == LEFT:
        makeMove(board, RIGHT)
    elif move == RIGHT:
        makeMove(board, LEFT)


def getValidMoves(board, prevMove=None):
    """Funkcja zwraca listę dopuszczalnych ruchów na przekazanej jej planszy;
    Jeżeli przekazano wartość prevMove, funkcja nie uwzględni przekazanego
    w tym parametrze ruchu w wynikowej liście"""
    

    blankx, blanky = findBlankSpace(board)

    validMoves = []
    if blanky != SIZE - 1 and prevMove != DOWN:
        # Puste miejsce nie znajduje się w dolnym wierszu
        validMoves.append(UP)

    if blankx != SIZE - 1 and prevMove != RIGHT:
        # Puste miejsce nie znajduje się w prawej kolumnie
        validMoves.append(LEFT)

    if blanky != 0 and prevMove != UP:
        # Puste miejsce nie znajduje się w pierwszym wierszu
        validMoves.append(DOWN)

    if blankx != 0 and prevMove != LEFT:
        # Puste miejsce nie znajduje się w lewej kolumnie
        validMoves.append(RIGHT)

    return validMoves



def getNewPuzzle():
    """Funkcja generuje nową układankę poprzez wykonanie losowych ruchów w rozwiązanej łamigłówce"""
    board = getNewBoard()
    for i in range(DIFFICULTY):
        validMoves = getValidMoves(board)
        makeMove(board, random.choice(validMoves))
    return board


def solve(board, maxMoves):
    """Próba znalezienia rozwiązania łamigłówki `board` składającego się z maksymalnie `maxMoves` ruchów;
    po znalezieniu rozwiązania funkcja zwraca True, w przeciwnym przypadku zwraca False"""
    print('Próbuję znaleźć rozwiązanie składające się z maksymalnie', maxMoves, 'ruchów...')
    solutionMoves = [] # Lista wartości UP, DOWN, LEFT, RIGHT
    solved = attemptMove(board, solutionMoves, maxMoves, None)

    if solved:
        displayBoard(board)
        for move in solutionMoves:
            print('Przesuwam klocek', move)
            makeMove(board, move)
            print() # Przejście do nowej linii
            displayBoard(board)
            print() # Przejście do nowej linii

        print('Rozwiązanie zawierające', len(solutionMoves), 'ruchów:')
        print(', '.join(solutionMoves))
        return True # Łamigłówka została rozwiązana
    else:
        return False # Brak rozwiązania składającego się z co najwyżej maxMoves ruchów


def attemptMove(board, movesMade, movesRemaining, prevMove):
    """Funkcja rekurencyjna, która sprawdza wszystkie dopuszczalne na planszy 'board'
    sekwencje ruchów o maksymalnej długości 'maxMoves'. Funkcja zwraca True
    po znalezieniu rozwiązania. W tym przypadku zmienna `movesMade` zawiera
    ciąg ruchów, które należy wykonać, aby rozwiązać zagadkę. Funkcja zwraca False
    gdy `movesRemaining` jest mniejsze niż 0"""

    if movesRemaining < 0:
        # PRZYPADEK BAZOWY — Przekroczono dopuszczalną liczbę ruchów
        return False

    if board == SOLVED_BOARD:
        # PRZYPADEK BAZOWY — Znaleziono rozwiązanie
        return True

    # PRZYPADEK REKURENCYJNY - Próba wykonania wszystkich dopuszczalnych ruchów:
    for move in getValidMoves(board, prevMove):
        # Wykonanie ruchu:
        makeMove(board, move)
        movesMade.append(move)

        if attemptMove(board, movesMade, movesRemaining - 1, move):
            # Jeżeli znaleziono rozwiązanie, funkcja zwraca True:
            undoMove(board, move) # Powrót do stanu początkowego
            return True

        # Cofamy ruch, aby przygotować się do wykonania kolejnego:
        undoMove(board, move)
        movesMade.pop() # Usuwanie ostatniego, cofniętego ruchu
    return False # PRZYPADEK BAZOWY - Nie udało się znaleźć rozwiązania


# Wykonanie programu:
SOLVED_BOARD = getNewBoard()
puzzleBoard = getNewPuzzle()
displayBoard(puzzleBoard)
startTime = time.time()

maxMoves = 10
while True:
    if solve(puzzleBoard, maxMoves):
        break # Wyjście z pętli po znalezieniu rozwiązania
    maxMoves += 1
print('Czas wykonania:', round(time.time() - startTime, 3), 'sekund')
